use util:serialize;
use core:io;
use parser;

/**
 * Misc. data types.
 */


/**
 * Representation of a single problem (no source code, intended for lists of problems).
 */
class ProblemInfo : serializable {
	Int id;

	Str title;
	Str author;

	// Is this my problem?
	Bool myProblem;

	// Is the next step to modify the implementation?
	Bool currentError;

	init(Int id, Str title, Str author, Bool myProblem, Bool currentError) {
		init {
			id = id;
			title = title;
			author = author;
			myProblem = myProblem;
			currentError = currentError;
		}
	}

	// For sorting.
	Bool <(ProblemInfo o) {
		id < o.id;
	}

	Str typeStr() {
		if (currentError)
			"Debug";
		else
			"Test";
	}
}

/**
 * A problem with improved and solved stats.
 */
class ProblemDetails : extends ProblemInfo, serializable {
	// Number of improvements.
	Nat improvements;

	// Number of solutions. Only relevant if 'currentError = true'
	Nat solutions;

	init(Int id, Str title, Str author, Bool myProblem, Bool currentError, Nat improvements, Nat solutions) {
		init(id, title, author, myProblem, currentError) {
			improvements = improvements;
			solutions = solutions;
		}
	}

	Str improvedStr() {
		if (currentError)
			solutions.toS + "+" + improvements.toS;
		else
			improvements.toS;
	}
}


/**
 * Representation of a problem, including code.
 */
class Problem : extends ProblemInfo, serializable {
	// Main program.
	Code main;

	// Implementation of the data structure.
	Code impl;

	// Reference implementation of the data structure.
	Code ref;

	// Create.
	init(Int id, Str title, Str author, Bool myProblem, Bool currentError, Code main, Code impl, Code ref) {
		init(id, title, author, myProblem, currentError) {
			main = main;
			impl = impl;
			ref = ref;
		}
	}
}

/**
 * A solution to a problem. I.e. a recorded sequence of steps that illustrates some error.
 */
class Solution : serializable {
	// ID of this solution.
	Int id;

	// Solution to what problem?
	Int problemId;

	// Author of the solution.
	Str author;

	// Type of error found.
	Str type;

	// Sequence of steps in the solution.
	Str steps;

	// Create.
	init(Int id, Int problemId, Str author, Str type, Str steps) {
		init { id = id; problemId = problemId; author = author; type = type; steps = steps; }
	}
}

/**
 * A piece of code, indicating a language as well.
 */
class Code : serializable {
	// Source code.
	Str src;

	// Language (= file extension).
	Str language;

	// Create.
	init(Str src, Str language) {
		init {
			src = src;
			language = language;
		}
	}

	// Add to a MemoryProtocol.
	Url put(Str base, MemoryProtocol to) {
		MemOStream out;
		Utf8Output text(out);
		text.write(src);
		text.flush();
		to.put(base + "." + language, out.buffer);
	}

	// Produce a "signature" for this code.
	Str signature() {
		codeSignature(src);
	}
}


/**
 * A single high-score entry.
 */
class Score : serializable {
	Str name;
	Int points;
	Int place;

	init(Str name, Int points, Int place) {
		init {
			name = name;
			points = points;
			place = place;
		}
	}
}

// Create a "signature" that allows comparing code for equality.
Str codeSignature(Str code) {
	sigParser(code).value.toS;
}

// Remove whitespace from a string to create a "hash" of a program. Note: This is not 100%
// accurate, but good enough to detect "significant" changes (for example "myclass a" and "my
// classa" will hash to the same thing.
sigParser : parser(recursive descent) {
	start = Start;

	StrBuf Start();
	Start => StrBuf() : "[ \n\r\t]*" - ("[^ \n\r\t]+" -> add - "[ \n\r\t]*")*;
}
